#ifndef _OGRENDERX_H
#define _OGRENDERX_H

// Copyright (c) OpenMedia Lab., 2003. ver.5/16/2004

#ifdef __gl_h__
#ifndef __GL_H__
#define __GL_H__
#endif
#endif

#ifdef __glut_h__
#ifndef __GLUT_H__
#define __GLUT_H__
#endif
#endif


#ifdef _WIN32
#ifdef __GL_H__
	#pragma comment (lib, "OpenGL32.lib")
	#pragma comment (lib, "GLU32.lib")
#endif

#ifdef _SDL_H
	#pragma comment (lib, "SDL.lib")
	#pragma comment (lib, "SDLmain.lib")
#endif

#pragma warning (disable:4244)
#endif

#include "geometry.h"
#include "bitmap.h"

//--------------------------------- transformation -------------------------------------
#ifdef __GL_H__
	#include <GL/glu.h>
#else
D3DXMATRIX _matrix[64];
int _currentMatrix = 0;
#endif

void ogPushMatrix(){
#ifdef __GL_H__
		glPushMatrix();
#else
	g_pApp->getDevice()->GetTransform( D3DTS_WORLD, &_matrix[_currentMatrix++] );
#endif
}

void ogPopMatrix(){
#ifdef __GL_H__
		glPopMatrix();
#else
	_currentMatrix--;
	g_pApp->getDevice()->SetTransform( D3DTS_WORLD, &_matrix[_currentMatrix] );
#endif
}

void ogMultMatrix(ogMatrix &matrix){
#ifdef __GL_H__
	#if defined(RealIsFloat)
		glMultMatrixf(matrix.e);
	#else
		glMultMatrixd(matrix.e);
	#endif
#else
	#if defined(RealIsFloat)
		g_pApp->getDevice()->MultiplyTransform( D3DTS_WORLD, &D3DXMATRIX(matrix.e) );
	#else
		float e[16];
		for(int i=0;i<16;i++) e[i] = matrix.e[i];
		g_pApp->getDevice()->MultiplyTransform( D3DTS_WORLD, &D3DXMATRIX(e) );
	#endif
#endif
}

void ogTranslate(float x, float y, float z){
#ifdef __GL_H__
	glTranslatef(x,y,z);
#else
	D3DXMATRIX matrix;
	D3DXMatrixTranslation(&matrix, x, y, z);
	g_pApp->getDevice()->MultiplyTransform( D3DTS_WORLD, &matrix );
#endif
}

void ogScale(float x, float y, float z){
#ifdef __GL_H__
	glScalef(x,y,z);
#else
	D3DXMATRIX matrix;
	D3DXMatrixScaling(&matrix, x, y, z);
	g_pApp->getDevice()->MultiplyTransform( D3DTS_WORLD, &matrix );
#endif
}

void ogRotate(float angle, float x, float y, float z){
#ifdef __GL_H__
	glRotatef( angle, x, y, z );
#else
	D3DXMATRIX matrix;
	D3DXMatrixRotationAxis(&matrix, &D3DXVECTOR3(x, y, z), rad(angle));
	g_pApp->getDevice()->MultiplyTransform( D3DTS_WORLD, &matrix );
#endif
}

//--------------------------------- misc -------------------------------------
void ogClear(unsigned char red, unsigned char green, unsigned char blue ){
#ifdef __GL_H__
	glClearColor( red/255.0, green/255.0, blue/255.0, 1.0 );
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
#else
	g_pApp->getDevice()->SetRenderState(D3DRS_CULLMODE, D3DCULL_NONE);
	g_pApp->getDevice()->Clear(0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(red, green, blue), 1.0f, 0);
	g_pApp->getDevice()->BeginScene();
#endif
}

void ogSwapBuffers(){
#ifdef __GL_H__
	#ifdef __GLUT_H__
		glutSwapBuffers();
	#else
		SDL_GL_SwapBuffers();
	#endif
#else
	g_pApp->getDevice()->EndScene();
	//g_pApp->getDevice()->Present(NULL, NULL, NULL, NULL);
#endif
}

void ogZBufferOn(){
#ifdef __GL_H__
	glEnable( GL_DEPTH_TEST );
#else
	g_pApp->getDevice()->SetRenderState( D3DRS_ZENABLE, D3DZB_TRUE );
#endif
}

void ogZBufferOff(){
#ifdef __GL_H__
	glDisable( GL_DEPTH_TEST );
#else
	g_pApp->getDevice()->SetRenderState( D3DRS_ZENABLE, D3DZB_FALSE );
#endif
}

void ogFlatShading(){
#ifdef __GL_H__
	glShadeModel( GL_FLAT );
#else
	g_pApp->getDevice()->SetRenderState( D3DRS_SHADEMODE, D3DSHADE_FLAT );
#endif
}

void ogGouraudShading(){
#ifdef __GL_H__
	glShadeModel( GL_SMOOTH );
#else
	g_pApp->getDevice()->SetRenderState( D3DRS_SHADEMODE, D3DSHADE_GOURAUD );
#endif
}

void ogLightPosition(float x, float y, float z){
#ifdef __GL_H__
	float position[4];
	position[0]=x;position[1]=y;position[2]=z;position[3]=0.0;
	glLightfv( GL_LIGHT0, GL_POSITION, position );
#else
	D3DLIGHT9 light;
	g_pApp->getDevice()->GetLight( 0, &light );
    D3DXVec3Normalize( (D3DXVECTOR3*)&light.Direction, &D3DXVECTOR3(-x, -y, -z) );
    light.Position.x   = x;
    light.Position.y   = y;
    light.Position.z   = z;
 	g_pApp->getDevice()->SetLight( 0, &light );
#endif
}

void ogLightOn(float x, float y, float z, float diffuseRed, float diffuseGreen, float diffuseBlue, float specularRed, float specularGreen, float specularBlue, float ambientRed, float ambientGreen, float ambientBlue, float shininess){
#ifdef __GL_H__
	glEnable( GL_LIGHTING );
	glEnable( GL_LIGHT0 );
	glEnable( GL_COLOR_MATERIAL );
	GLfloat diffuse[]={diffuseRed, diffuseGreen, diffuseBlue, 1.0f};
	GLfloat specular[]={specularRed, specularGreen, specularBlue, 1.0f};
	GLfloat ambient[]={ambientRed, ambientGreen, ambientBlue, 1.0f};
	glLightfv( GL_LIGHT0, GL_DIFFUSE, diffuse );
	glLightfv( GL_LIGHT0, GL_SPECULAR, specular );
	glLightfv( GL_LIGHT0, GL_AMBIENT, ambient );
	ogLightPosition(x,y,z);
	GLfloat mat_specular[]={1.0f, 1.0f, 1.0f, 1.0f};
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shininess);
#else
	D3DLIGHT9 light;
	ZeroMemory( &light, sizeof(D3DLIGHT9) );
    light.Type        = D3DLIGHT_DIRECTIONAL;
    light.Diffuse.r   = diffuseRed;
    light.Diffuse.g   = diffuseGreen;
    light.Diffuse.b   = diffuseBlue;
    light.Diffuse.a   = 1.0;
    light.Specular.r  = specularRed;
    light.Specular.g  = specularGreen;
    light.Specular.b  = specularBlue;
    light.Specular.a  = 1.0;
    light.Ambient.r   = ambientRed;
    light.Ambient.g   = ambientGreen;
    light.Ambient.b   = ambientBlue;
    light.Ambient.a   = 1.0;
   D3DXVec3Normalize( (D3DXVECTOR3*)&light.Direction, &D3DXVECTOR3(-x, -y, -z) );
    light.Position.x   = x;
    light.Position.y   = y;
    light.Position.z   = z;
    light.Range        = 1000.0f;
	g_pApp->getDevice()->SetLight( 0, &light );
	g_pApp->getDevice()->SetRenderState(D3DRS_SPECULARENABLE,TRUE);
	g_pApp->getDevice()->SetRenderState(D3DRS_AMBIENTMATERIALSOURCE,D3DMCS_COLOR1);
//	g_pApp->getDevice()->SetRenderState( D3DRS_AMBIENT, D3DCOLOR_XRGB(255,255,255));
	g_pApp->getDevice()->LightEnable( 0, TRUE );
	D3DMATERIAL9 material;
	ZeroMemory( &material, sizeof(D3DMATERIAL9) );
    material.Diffuse.r = diffuseRed;
    material.Diffuse.g   = diffuseGreen;
    material.Diffuse.b   = diffuseBlue;
    material.Diffuse.a   = 1.0;
    material.Specular.r  = specularRed;
    material.Specular.g  = specularGreen;
    material.Specular.b  = specularBlue;
    material.Specular.a  = 1.0;
    material.Ambient.r   = ambientRed;
    material.Ambient.g   = ambientGreen;
    material.Ambient.b   = ambientBlue;
    material.Ambient.a   = 1.0;
	material.Power = shininess;
	g_pApp->getDevice()->SetMaterial( &material );
	g_pApp->getDevice()->SetRenderState( D3DRS_LIGHTING, TRUE );
#endif
}

void ogLightOff(){
#ifdef __GL_H__
	glDisable( GL_LIGHTING );
#else
	g_pApp->getDevice()->SetRenderState( D3DRS_LIGHTING, FALSE );
#endif
}

void ogViewport(int x, int y, int width, int height){
#ifdef __GL_H__
	glViewport( x, y, width, height );
#else
	D3DVIEWPORT9 vp;
	vp.X		= x;
	vp.Y		= y;
	vp.Width	= width;
	vp.Height	= height;
	vp.MinZ		= 0.0f;
	vp.MaxZ		= 1.0f;
	if( g_pApp->getDevice() ) g_pApp->getDevice()->SetViewport(&vp);
#endif
}

void ogOrtho2D( float left, float right, float bottom, float top ){
#ifdef __GL_H__
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	gluOrtho2D( left, right, bottom, top );
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();
#else
	D3DXMATRIX matrix;
	D3DXMatrixOrthoOffCenterLH( &matrix, left, right, bottom, top, -1.0f, 1.0f );
	g_pApp->getDevice()->SetTransform( D3DTS_PROJECTION, &matrix );
#endif
}

void ogLookAt(float ex, float ey, float ez, float cx, float cy, float cz, float ux, float uy, float uz){
#ifdef __GL_H__
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();
	gluLookAt( ex, ey, ez, cx, cy, cz, ux, uy, uz );
#else
	D3DXMATRIX matrix;
	D3DXMatrixLookAtRH(&matrix,&D3DXVECTOR3(ex,ey,ez),&D3DXVECTOR3(cx,cy,cz),&D3DXVECTOR3(ux,uy,uz));
	g_pApp->getDevice()->SetTransform( D3DTS_VIEW, &matrix );
	D3DXMatrixIdentity(&matrix);
	g_pApp->getDevice()->SetTransform( D3DTS_WORLD, &matrix );
#endif
}

void ogGetViewport(int &x, int &y, int &width, int &height){
#ifdef __GL_H__
	GLint v[4];
	glGetIntegerv( GL_VIEWPORT, v );
	x = v[0];
	y = v[1];
	width = v[2];
	height = v[3];
#else
	D3DVIEWPORT9 vp;
	g_pApp->getDevice()->GetViewport( &vp );
	x = vp.X;
	y = vp.Y;
	width = vp.Width;
	height = vp.Height;
#endif
}

void ogRotateEyePosition( int mouseX, int mouseY ){
	int x, y, width, height;
	ogGetViewport( x, y, width, height );
	int xd = mouseX - width/2;
	int yd = -(mouseY - height/2);
	if( xd != 0 || yd != 0 ){
		#ifdef __GL_H__
		ogRotate( sqrt((double)xd*xd+yd*yd), -yd, xd, 0 );
		#else
		D3DXMATRIX matrix;
		D3DXMatrixRotationAxis(&matrix, &D3DXVECTOR3(-yd, xd, 0), rad(sqrt((double)xd*xd+yd*yd)));
		g_pApp->getDevice()->MultiplyTransform( D3DTS_VIEW, &matrix );
		#endif
	}
}

void ogPerspective(float fovy, float nearClip, float farClip){
	int x, y, width, height;
	ogGetViewport(x,y,width,height);
#ifdef __GL_H__
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	gluPerspective( fovy, GLdouble(width)/GLdouble(height), nearClip, farClip );
	glMatrixMode( GL_MODELVIEW );
#else
	D3DXMATRIX matrix;
	D3DXMatrixPerspectiveFovRH(&matrix,D3DXToRadian(fovy),float(width)/float(height), nearClip,farClip);
	g_pApp->getDevice()->SetTransform(D3DTS_PROJECTION,&matrix);
#endif
}

void ogWindow2World( Double windowX, Double windowY, Double windowZ, Double &worldX, Double &worldY, Double &worldZ ){
#ifdef __GL_H__
	GLdouble mm[16], pm[16];
	GLint v[4];
	glGetDoublev( GL_MODELVIEW_MATRIX,  mm );
	glGetDoublev( GL_PROJECTION_MATRIX, pm );
	glGetIntegerv( GL_VIEWPORT, v );
	double wx,wy,wz;
	gluUnProject( windowX, Double(v[3])-windowY, windowZ, mm, pm, v, &wx, &wy, &wz );
	worldX = wx;
	worldY = wy;
	worldZ = wz;
#else
	D3DVIEWPORT9 viewport;
	D3DXMATRIX projectionMatrix, viewMatrix, worldMatrix;
	D3DXVECTOR3 window, world;

	window.x = float(windowX);
	window.y = float(windowY);
	window.z = float(windowZ);
	g_pApp->getDevice()->GetViewport( &viewport );
	g_pApp->getDevice()->GetTransform( D3DTS_PROJECTION, &projectionMatrix );
	g_pApp->getDevice()->GetTransform( D3DTS_VIEW, &viewMatrix );
	g_pApp->getDevice()->GetTransform( D3DTS_WORLD, &worldMatrix );
	D3DXVec3Unproject( &world, &window, &viewport, &projectionMatrix, &viewMatrix, &worldMatrix );
	worldX = world.x;
	worldY = world.y;
	worldZ = world.z;
#endif
}

void ogWindow2World2D( Double windowX, Double windowY, Double &worldX, Double &worldY ){
	Double worldZ;
	ogWindow2World( windowX, windowY, 0.0, worldX, worldY, worldZ );
}

void ogWorld2Window( Double worldX, Double worldY, Double worldZ, Double &windowX, Double &windowY, Double &windowZ ){
#ifdef __GL_H__
	GLdouble mm[16], pm[16];
	GLint v[4];
	glGetDoublev( GL_MODELVIEW_MATRIX,  mm );
	glGetDoublev( GL_PROJECTION_MATRIX, pm );
	glGetIntegerv( GL_VIEWPORT, v );
	double wx,wy,wz;
	gluProject( worldX, worldY, worldZ, mm, pm, v, &wx, &wy, &wz );
	windowX = wx;
	windowY = v[3] - wy;
	windowZ = wz;
#else
	D3DVIEWPORT9 viewport;
	D3DXMATRIX projectionMatrix, viewMatrix, worldMatrix;
	D3DXVECTOR3 window, world;

	world.x = float(worldX);
	world.y = float(worldY);
	world.z = float(worldZ);
	g_pApp->getDevice()->GetViewport( &viewport );
	g_pApp->getDevice()->GetTransform( D3DTS_PROJECTION, &projectionMatrix );
	g_pApp->getDevice()->GetTransform( D3DTS_VIEW, &viewMatrix );
	g_pApp->getDevice()->GetTransform( D3DTS_WORLD, &worldMatrix );
	D3DXVec3Project( &window, &world, &viewport, &projectionMatrix, &viewMatrix, &worldMatrix );
	windowX = window.x;
	windowY = window.y;
	windowZ = window.z;
#endif
}

void ogWorld2Window2D( Double worldX, Double worldY, Double &windowX, Double &windowY ){
	Double windowZ;
	ogWorld2Window( worldX, worldY, 0.0, windowX, windowY, windowZ );
}

void ogRedisplay(){
#ifdef __GL_H__
	glutPostRedisplay();
#else
#endif
}

void ogTextureOn(){
#ifdef __GL_H__
	glEnable( GL_TEXTURE_2D );
#else
	g_pApp->getDevice()->SetTextureStageState(0,D3DTSS_COLOROP, D3DTOP_MODULATE);
#endif
}

void ogTextureOff(){
#ifdef __GL_H__
	glDisable( GL_TEXTURE_2D );
#else
	g_pApp->getDevice()->SetTextureStageState(0,D3DTSS_COLOROP, D3DTOP_DISABLE);
#endif
}

//--------------------------------- drawing -------------------------------------
class ogVFxyzc{
public:
	float x,y,z;
	unsigned char b;
	unsigned char g;
	unsigned char r;
	unsigned char a;
	ogVFxyzc(){ z = 0.0; a = 255; }
	void set(float x, float y, float z){ this->x=x; this->y=y; this->z=z; }
	void set(float x, float y){ this->x=x; this->y=y; }
	void set(ogVector &v){ this->x=v.x; this->y=v.y; this->z=v.z; }
	void set(ogVectorf &v){ this->x=v.x; this->y=v.y; this->z=v.z; }
	void setColor(unsigned char r, unsigned char g, unsigned char b){ this->r=r; this->g=g; this->b=b; }

};

class ogVFxyzct:public ogVFxyzc{
public:
	float u, v;
	ogVFxyzct():ogVFxyzc(){ }
	void setTexCoord(float u, float v){ this->u=u; this->v=v; }
	void setTexCoord(ogVector2 &uv){ u=uv.x; v=uv.y; }
};

#ifdef __GL_H__
void _glVxyzc(ogVFxyzc vertex[], int n){
	for(int i=0; i<n; i++){
		glColor3ub( vertex[i].r, vertex[i].g, vertex[i].b );
		glVertex3f( vertex[i].x, vertex[i].y, vertex[i].z );
	}
}
void _glVxyzct(ogVFxyzct vertex[], int n){
	for(int i=0; i<n; i++){
		glColor3ub( vertex[i].r, vertex[i].g, vertex[i].b );
		glTexCoord2f( vertex[i].u, vertex[i].v );
		glVertex3f( vertex[i].x, vertex[i].y, vertex[i].z );
	}
}
#endif

void ogDrawPolygon(ogVFxyzc vertex[], int n){
#ifdef __GL_H__
	glBegin(GL_POLYGON);
	_glVxyzc(vertex, n);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_DIFFUSE);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_TRIANGLEFAN,n-2,vertex,sizeof(ogVFxyzc));
#endif
}

void ogDrawTexturePolygon(ogVFxyzct vertex[], int n){
	ogTextureOn();
#ifdef __GL_H__
	glBegin(GL_POLYGON);
	_glVxyzct(vertex, n);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_DIFFUSE|D3DFVF_TEX1);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_TRIANGLEFAN,n-2,vertex,sizeof(ogVFxyzct));
#endif
	ogTextureOff();
}

void ogDrawTriangle(Double x1, Double y1, Double z1, unsigned char r1, unsigned char g1, unsigned char b1,
                    Double x2, Double y2, Double z2, unsigned char r2, unsigned char g2, unsigned char b2,
                    Double x3, Double y3, Double z3, unsigned char r3, unsigned char g3, unsigned char b3){
	ogVFxyzc v[3];
	v[0].set(x1,y1,z1);
	v[1].set(x2,y2,z2);
	v[2].set(x3,y3,z3);
	v[0].setColor(r1,g1,b1);
	v[1].setColor(r2,g2,b2);
	v[2].setColor(r3,g3,b3);
	ogDrawPolygon(v,3);
}

void ogDrawQuad(Double x1, Double y1, Double z1, unsigned char r1, unsigned char g1, unsigned char b1,
                Double x2, Double y2, Double z2, unsigned char r2, unsigned char g2, unsigned char b2,
                Double x3, Double y3, Double z3, unsigned char r3, unsigned char g3, unsigned char b3,
                Double x4, Double y4, Double z4, unsigned char r4, unsigned char g4, unsigned char b4){
	ogVFxyzc v[4];
	v[0].set(x1,y1,z1);
	v[1].set(x2,y2,z2);
	v[2].set(x3,y3,z3);
	v[3].set(x4,y4,z4);
	v[0].setColor(r1,g1,b1);
	v[1].setColor(r2,g2,b2);
	v[2].setColor(r3,g3,b3);
	v[3].setColor(r4,g4,b4);
	ogDrawPolygon(v,4);
}

void ogDrawTextureTriangle(Double x1, Double y1, Double z1, unsigned char r1, unsigned char g1, unsigned char b1, Double u1, Double v1,
                           Double x2, Double y2, Double z2, unsigned char r2, unsigned char g2, unsigned char b2, Double u2, Double v2,
                           Double x3, Double y3, Double z3, unsigned char r3, unsigned char g3, unsigned char b3, Double u3, Double v3){
	ogVFxyzct v[3];
	v[0].set(x1,y1,z1);
	v[1].set(x2,y2,z2);
	v[2].set(x3,y3,z3);
	v[0].setColor(r1,g1,b1);
	v[1].setColor(r2,g2,b2);
	v[2].setColor(r3,g3,b3);
	v[0].setTexCoord(u1,v1);
	v[1].setTexCoord(u2,v2);
	v[2].setTexCoord(u3,v3);
	ogDrawTexturePolygon(v,3);
}

void ogDrawTextureQuad(Double x1, Double y1, Double z1, unsigned char r1, unsigned char g1, unsigned char b1, Double u1, Double v1,
                       Double x2, Double y2, Double z2, unsigned char r2, unsigned char g2, unsigned char b2, Double u2, Double v2,
                       Double x3, Double y3, Double z3, unsigned char r3, unsigned char g3, unsigned char b3, Double u3, Double v3,
                       Double x4, Double y4, Double z4, unsigned char r4, unsigned char g4, unsigned char b4, Double u4, Double v4){
	ogVFxyzct v[4];
	v[0].set(x1,y1,z1);
	v[1].set(x2,y2,z2);
	v[2].set(x3,y3,z3);
	v[3].set(x4,y4,z4);
	v[0].setColor(r1,g1,b1);
	v[1].setColor(r2,g2,b2);
	v[2].setColor(r3,g3,b3);
	v[3].setColor(r4,g4,b4);
	v[0].setTexCoord(u1,v1);
	v[1].setTexCoord(u2,v2);
	v[2].setTexCoord(u3,v3);
	v[3].setTexCoord(u4,v4);
	ogDrawTexturePolygon(v,4);
}

void ogDrawLines(ogVFxyzc vertex[], int n){
#ifdef __GL_H__
	glBegin(GL_LINES);
	_glVxyzc(vertex, n);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_DIFFUSE);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_LINELIST,n/2,vertex,sizeof(ogVFxyzc));
#endif
}

void ogDrawLineStrip(ogVFxyzc vertex[], int n){
#ifdef __GL_H__
	glBegin(GL_LINE_STRIP);
	_glVxyzc(vertex, n);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_DIFFUSE);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_LINESTRIP,n-1,vertex,sizeof(ogVFxyzc));
#endif
}

void ogDrawLine(Double x1, Double y1, Double z1, unsigned char r1, unsigned char g1, unsigned char b1,
				Double x2, Double y2, Double z2, unsigned char r2, unsigned char g2, unsigned char b2){
	ogVFxyzc v[2];
	v[0].set(x1,y1,z1);
	v[1].set(x2,y2,z2);
	v[0].setColor(r1,g1,b1);
	v[1].setColor(r2,g2,b2);
	ogDrawLines(v,2);
}

class ogPolygonModel{
public:
	ogVFxyzc *v;
	int n;
	float x, y;
	unsigned char red, green, blue;
	void construct(int n){
		v = new ogVFxyzc[this->n = n];
		x = y = 0.0;
	}
	ogPolygonModel(unsigned char red, unsigned char green, unsigned char blue, int n){
		this->construct(n);
		this->setColor(red, green, blue);
	}
	ogPolygonModel(int n){ this->construct(n); }
	~ogPolygonModel(){ delete v; }
	void draw(float x, float y){
		this->x = x;
		this->y = y;
		ogPushMatrix();
		ogTranslate(x,y,0.0);
		ogDrawPolygon(v,n);
		ogPopMatrix();
	}
	void draw(float x, float y, float scale){
		this->x = x;
		this->y = y;
		ogPushMatrix();
		ogTranslate(x,y,0.0);
		ogScale(scale,scale,scale);
		ogDrawPolygon(v,n);
		ogPopMatrix();
	}
	void setColor(unsigned char red, unsigned char green, unsigned char blue){ this->red=red; this->green=green; this->blue=blue; for(int i=0; i<n; i++) v[i].setColor(red, green, blue); }
};

class ogRectangleModel:public ogPolygonModel{
public:
	float w, h;
	void setVertex(float w, float h){
		this->w = w;
		this->h = h;
		v[0].set(-0.5*w, -0.5*h);
		v[1].set(-0.5*w,  0.5*h);
		v[2].set( 0.5*w,  0.5*h);
		v[3].set( 0.5*w, -0.5*h);
	}
	ogRectangleModel(float w, float h, unsigned char red, unsigned char green, unsigned char blue):ogPolygonModel(red,green,blue,4){
		this->setVertex(w,h);
	}
	ogRectangleModel():ogPolygonModel(4){ }
	void draw(float x, float y){ ogPolygonModel::draw(x,y); }
	void draw(float x, float y, float w, float h){
		this->setVertex(w,h);
		this->draw(x,y);
	}
	void draw(float x, float y, float w, float h, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,w,h);
	}
};

void ogDrawRectangle(float x, float y, float w, float h, unsigned char red, unsigned char green, unsigned char blue){
	static ogRectangleModel model;
	model.draw(x,y,w,h,red, green, blue);
}

class ogCircleModel:public ogPolygonModel{
public:
	float r;
	float *px;
	float *py;
	void setVertex(float r){
		this->r = r;
		for(int i=0; i<n; i++ ) v[i].set(px[i]*r, py[i]*r);
	}
	void construct(int n = 16){
		px = new float[n];
		py = new float[n];
		for(int i=0; i<n; i++ ){
			v[i].x = px[i] = cos(rad(360.0*i/n));
			v[i].y = py[i] = sin(rad(360.0*i/n));
		}
	}
	ogCircleModel(float r, unsigned char red, unsigned char green, unsigned char blue, int n = 16):ogPolygonModel(red,green,blue,n){
		this->construct(n);
		this->setVertex(r);
	}
	ogCircleModel(int n = 16):ogPolygonModel(n){ this->construct(n); }
	~ogCircleModel(){ delete px; delete py; }
	void draw(float x, float y){ ogPolygonModel::draw(x,y); }
	void draw(float x, float y, float r){
		//this->setVertex(r);
		ogPolygonModel::draw(x,y,r);
	}
	void draw(float x, float y, float r, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,r);
	}
};

void ogDrawCircle(float x, float y, float r, unsigned char red, unsigned char green, unsigned char blue, int n){
	static ogCircleModel model(n);
	model.draw(x,y,r,red,green,blue);
}

class ogEllipseModel:public ogPolygonModel{
public:
	float a,b;
	float *px;
	float *py;
	void setVertex(float a,float b){
		this->a = a;
		this->b = b;
		for(int i=0; i<n; i++ ) v[i].set(px[i]*a, py[i]*b);
	}
	void construct(int n = 16){
		px = new float[n];
		py = new float[n];
		for(int i=0; i<n; i++ ){
			px[i] = cos(rad(360.0*i/n));
			py[i] = sin(rad(360.0*i/n));
		}
	}
	ogEllipseModel(float a, float b, unsigned char red, unsigned char green, unsigned char blue, int n = 16):ogPolygonModel(red,green,blue,n){
		this->construct(n);
		this->setVertex(a,b);
	}
	ogEllipseModel(int n = 16):ogPolygonModel(n){ this->construct(n); }
	~ogEllipseModel(){ delete px; delete py; }
	void draw(float x, float y){ ogPolygonModel::draw(x,y); }
	void draw(float x, float y, float a, float b){
		this->setVertex(a,b);
		this->draw(x,y);
	}
	void draw(float x, float y, float a, float b, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,a,b);
	}
};

void ogDrawEllipse(float x, float y, float a, float b, unsigned char red, unsigned char green, unsigned char blue, int n){
	static ogEllipseModel model(n);
	model.draw(x,y,a,b,red,green,blue);
}



class ogVFxyznc{
public:
	float x,y,z;
	float nx,ny,nz;
	unsigned char b;
	unsigned char g;
	unsigned char r;
	unsigned char a;
	ogVFxyznc(){ z = 0.0; a = 255; }
	void set(float x, float y, float z){ this->x=x; this->y=y; this->z=z; }
	void set(float x, float y){ this->x=x; this->y=y; }
	void set(ogVector &v){ this->x=v.x; this->y=v.y; this->z=v.z; }
	void setNormal(float nx, float ny, float nz){ this->nx=nx; this->ny=ny; this->nz=nz; }
	void setNormal(ogVector &nv){ this->nx=nv.x; this->ny=nv.y; this->nz=nv.z; }
	void setColor(unsigned char r, unsigned char g, unsigned char b){ this->r=r; this->g=g; this->b=b; }

};

#ifdef __GL_H__
void _glVxyznc(ogVFxyznc vertex[], int n){
	for(int i=0; i<n; i++){
		glColor3ub( vertex[i].r, vertex[i].g, vertex[i].b );
		glNormal3f( vertex[i].nx, vertex[i].ny, vertex[i].nz );
		glVertex3f( vertex[i].x, vertex[i].y, vertex[i].z );
	}
}
#endif

void ogDrawTriangles(ogVFxyznc vertex[], int n){
#ifdef __GL_H__
	glBegin(GL_TRIANGLES);
	_glVxyznc(vertex, n*3);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_NORMAL | D3DFVF_DIFFUSE);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_TRIANGLELIST,n,vertex,sizeof(ogVFxyznc));
#endif
}

void ogDrawTriangleStrip(ogVFxyznc vertex[], int n){
#ifdef __GL_H__
	glBegin(GL_TRIANGLE_STRIP);
	_glVxyznc(vertex, n+2);
	glEnd();
#else
	g_pApp->getDevice()->SetFVF(D3DFVF_XYZ | D3DFVF_NORMAL | D3DFVF_DIFFUSE);
	g_pApp->getDevice()->DrawPrimitiveUP(D3DPT_TRIANGLESTRIP,n,vertex,sizeof(ogVFxyznc));
#endif
}

class ogPatchModel{
public:
	ogVFxyznc *v;
	int n;
	float x, y, z;
	unsigned char red, green, blue;
	void construct(int n){
		v = new ogVFxyznc[(this->n = n)*3];
		x = y = z = 0.0;
	}
	ogPatchModel(unsigned char red, unsigned char green, unsigned char blue, int n){
		this->construct(n);
		this->setColor(red, green, blue);
	}
	ogPatchModel(int n){ this->construct(n); }
	~ogPatchModel(){ delete v; }
	void draw(float x, float y, float z){
		this->x = x;
		this->y = y;
		this->z = z;
		ogPushMatrix();
		ogTranslate(x,y,z);
		ogDrawTriangles(v,n);
		ogPopMatrix();
	}
	void setColor(unsigned char red, unsigned char green, unsigned char blue){ this->red=red; this->green=green; this->blue=blue; for(int i=0; i<n*3; i++) v[i].setColor(red, green, blue); }
};

class ogBlockModel:public ogPatchModel{
public:
	float a, b, c;
	void setQuad(int n0, float x1, float y1, float z1, float x2, float y2, float z2, float x3, float y3, float z3, float x4, float y4, float z4, float nx, float ny, float nz ){
		v[n0].set(x1, y1, z1);
		v[n0+1].set(x2, y2, z2);
		v[n0+2].set(x4, y4, z4);
		v[n0+3].set(x2, y2, z2);
		v[n0+4].set(x3, y3, z3);
		v[n0+5].set(x4, y4, z4);
		for(int i=0; i<6; i++ ) v[n0+i].setNormal(nx,ny,nz);
	}
	void setVertex(float a, float b, float c){
		this->a = a;
		this->b = b;
		this->c = c;
		float a2 = a*0.5;
		float b2 = b*0.5;
		float c2 = c*0.5;
		this->setQuad(  0, -a2, -b2, -c2,  -a2,  b2, -c2,  a2,  b2, -c2,  a2, -b2, -c2, 0, 0, -1 );
		this->setQuad(  6, -a2, -b2,  c2,   a2, -b2,  c2,  a2,  b2,  c2, -a2,  b2,  c2, 0, 0,  1 );
		this->setQuad( 12, -a2, -b2, -c2,  -a2, -b2,  c2, -a2,  b2,  c2, -a2,  b2, -c2, -1, 0, 0 );
		this->setQuad( 18,  a2, -b2, -c2,   a2,  b2, -c2,  a2,  b2,  c2,  a2, -b2,  c2,  1, 0, 0 );
		this->setQuad( 24, -a2, -b2, -c2,   a2, -b2, -c2,  a2, -b2,  c2, -a2, -b2,  c2, 0, -1, 0 );
		this->setQuad( 30, -a2,  b2, -c2,  -a2,  b2,  c2,  a2,  b2,  c2,  a2,  b2, -c2, 0,  1, 0 );
	}
	ogBlockModel(float a, float b, float c, unsigned char red, unsigned char green, unsigned char blue):ogPatchModel(red,green,blue,12){
		this->setVertex(a,b,c);
	}
	ogBlockModel():ogPatchModel(12){ }
	void draw(float x, float y, float z){ ogPatchModel::draw(x,y,z); }
	void draw(float x, float y, float z, float a, float b, float c){
		this->setVertex(a,b,c);
		this->draw(x,y,z);
	}
	void draw(float x, float y, float z, float a, float b, float c, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,z,a,b,c);
	}
};

void ogDrawBlock(float x, float y, float z, float a, float b, float c, unsigned char red, unsigned char green, unsigned char blue){
	static ogBlockModel model;
	model.draw(x,y,z,a,b,c,red,green,blue);
}

class ogCylinderModel:public ogPatchModel{
public:
	float r, h;
	int n2;
	float *px;
	float *py;
	void setVertex(float r, float h){
		this->r = r;
		this->h = h;
		float h2 = h*0.5;
		for(int i=0; i<n2; i++){
			v[i*6].set(r*px[i],r*py[i],-h2);
			v[i*6+1].set(r*px[i],r*py[i],h2);
			v[i*6+2].set(r*px[i+1],r*py[i+1],-h2);
			v[i*6+3].set(r*px[i],r*py[i],h2);
			v[i*6+4].set(r*px[i+1],r*py[i+1],h2);
			v[i*6+5].set(r*px[i+1],r*py[i+1],-h2);
			v[i*6].setNormal(px[i],py[i],0.0);
			v[i*6+1].setNormal(px[i],py[i],0.0);
			v[i*6+2].setNormal(px[i+1],py[i+1],0.0);
			v[i*6+3].setNormal(px[i],py[i],0.0);
			v[i*6+4].setNormal(px[i+1],py[i+1],0.0);
			v[i*6+5].setNormal(px[i+1],py[i+1],0.0);
		}
	}
	void construct(int n = 16){
		n2 = n;
		px = new float[n2+1];
		py = new float[n2+1];
		for(int i=0; i<n2+1; i++ ){
			px[i] = cos(rad(360.0*i/n2));
			py[i] = sin(rad(360.0*i/n2));
		}
	}
	ogCylinderModel(float r, float h, unsigned char red, unsigned char green, unsigned char blue, int n = 16):ogPatchModel(red,green,blue,n*2){
		this->construct(n);
		this->setVertex(r,h);
	}
	ogCylinderModel(int n = 16):ogPatchModel(n*2){ this->construct(n); }
	~ogCylinderModel(){ delete px; delete py; }
	void draw(float x, float y, float z){ ogPatchModel::draw(x,y,z); }
	void draw(float x, float y, float z, float r, float h){
		this->setVertex(r,h);
		this->draw(x,y,z);
	}
	void draw(float x, float y, float z, float r, float h, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,z,r,h);
	}
};

void ogDrawCylinder(float x1, float y1, float z1, float x2, float y2, float z2, float r, unsigned char red, unsigned char green, unsigned char blue, int n){
	static ogCylinderModel model(n);
	ogVector v1(x1,y1,z1);
	ogVector v2(x2,y2,z2);
	ogVector v, o;
	if(z1<z2){ v = v2-v1; o=v1; }else{ v = v1 - v2; o=v2; }
	Double h=v.abs();
	v.normalize();
	ogPushMatrix();
	ogTranslate(o.x,o.y,o.z);
	ogVector a(v.y,-v.x,0.0);
	Double k=1.0/(1.0+v.z);
	ogMatrix matrix(
		k*a.x*a.x+v.z, k*a.x*a.y-a.z, k*a.x*a.z+a.y, 0.0,
		k*a.y*a.x+a.z, k*a.y*a.y+v.z, k*a.y*a.z-a.x, 0.0,
		k*a.z*a.x-a.y, k*a.z*a.y+a.x, k*a.z*a.z+v.z, 0.0,
		0.0, 0.0, 0.0, 1.0 );
	ogMultMatrix( matrix );
	model.draw(0.0,0.0,h/2,r,h,red,green,blue);
	ogPopMatrix();
}

class ogUnitSphere{
public:
	int      vertexSize;
	ogVector *vertex;
	int      faceSize;
	int     *index;
	void construct( int level=1 ); // level = 0, 1, 2
	ogUnitSphere( int level=1 ){ this->construct( level ); }
	~ogUnitSphere(){ delete vertex; delete index; }
private:
	void setIndex( int faceNumber, int a, int b, int c ){ index[faceNumber*3]=a; index[faceNumber*3+1]=b; index[faceNumber*3+2]=c; }
	int  newVertex( int index1, int index2 );
	void divideFace( int *index1, int *index2 );
};

void ogUnitSphere::construct( int level ){
	switch( level ){
		case 0: vertexSize = 12; faceSize = 20; break;
		case 1: vertexSize = 42; faceSize = 80; break;
		case 2: vertexSize = 162; faceSize = 320; break;
	}
	vertex = new ogVector[vertexSize];
	index = new int[faceSize*3];
	//make level 0
	Double sphereX=0.525731112119133606;
	Double sphereZ=0.850650808352039932;
	vertex[0].set( -sphereX, 0.0,  sphereZ );
	vertex[1].set(  sphereX, 0.0,  sphereZ );
	vertex[2].set( -sphereX, 0.0, -sphereZ );
	vertex[3].set(  sphereX, 0.0, -sphereZ );
	vertex[4].set( 0.0,  sphereZ,  sphereX );
	vertex[5].set( 0.0,  sphereZ, -sphereX );
	vertex[6].set( 0.0, -sphereZ,  sphereX );
	vertex[7].set( 0.0, -sphereZ, -sphereX );
	vertex[8].set(  sphereZ,  sphereX, 0.0 );
	vertex[9].set( -sphereZ,  sphereX, 0.0 );
	vertex[10].set(  sphereZ, -sphereX, 0.0 );
	vertex[11].set( -sphereZ, -sphereX, 0.0 );
	this->setIndex( 0, 0, 4, 1 );
	this->setIndex( 1, 0, 9, 4 );
	this->setIndex( 2, 9, 5, 4 );
	this->setIndex( 3, 4, 5, 8 );
	this->setIndex( 4, 4, 8, 1 );
	this->setIndex( 5, 8, 10, 1 );
	this->setIndex( 6, 8, 3, 10 );
	this->setIndex( 7, 5, 3, 8 );
	this->setIndex( 8, 5, 2, 3 );
	this->setIndex( 9, 2, 7, 3 );
	this->setIndex( 10, 7, 10, 3 );
	this->setIndex( 11, 7, 6, 10 );
	this->setIndex( 12, 7, 11, 6 );
	this->setIndex( 13, 11, 0, 6 );
	this->setIndex( 14, 0, 1, 6 );
	this->setIndex( 15, 6, 1, 10 );
	this->setIndex( 16, 9, 0, 11 );
	this->setIndex( 17, 9, 11, 2 );
	this->setIndex( 18, 9, 2, 5 );
	this->setIndex( 19, 7, 2, 11 );
	vertexSize = 12;
	//make level 1
	if( level >= 1 ){
		int index2[60];
		int i;
		for( i=0; i<60; i++ ) index2[i] = index[i];
		for( i=0; i<20; i++ ) divideFace( &index2[i*3], &index[i*12] );
	}
	//make level 2
	if( level == 2 ){
		int index2[240];
		int i;
		for( i=0; i<240; i++ ) index2[i] = index[i];
		for( i=0; i<80; i++ ) divideFace( &index2[i*3], &index[i*12] );
	}
}

int ogUnitSphere::newVertex( int index1, int index2 ){
ogVector v;
	v.add( vertex[index1], vertex[index2] );
	v.normalize();
	for( int i=12; i<vertexSize; i++ ){
		if( v.distance2( vertex[i] ) < 0.0000001 ) return i;
	}
	vertex[vertexSize++] = v;
	return vertexSize - 1;
}

void ogUnitSphere::divideFace( int *index1, int *index2 ){
	index2[0] = index1[0];
	index2[1] = this->newVertex( index1[0], index1[1] );
	index2[2] = this->newVertex( index1[0], index1[2] );
	index2[3] = index1[1];
	index2[4] = this->newVertex( index1[1], index1[2] );
	index2[5] = this->newVertex( index1[1], index1[0] );
	index2[6] = index1[2];
	index2[7] = this->newVertex( index1[2], index1[0] );
	index2[8] = this->newVertex( index1[2], index1[1] );
	index2[9] = this->newVertex( index1[0], index1[1] );
	index2[10] = this->newVertex( index1[1], index1[2] );
	index2[11] = this->newVertex( index1[2], index1[0] );
}

class ogSphereModel:public ogPatchModel{
public:
	float r;
	ogUnitSphere *sphere;
	void setVertex(float r){
		this->r = r;
		for(int i=0; i<sphere->faceSize*3; i++){
			ogVector v1 = r*sphere->vertex[sphere->index[i]];
			v[i].set(v1);
			v[i].setNormal(sphere->vertex[sphere->index[i]]);
		}
	}
	void construct(int level = 1){ sphere = new ogUnitSphere(level); }
	ogSphereModel(float r, unsigned char red, unsigned char green, unsigned char blue, int level = 1):ogPatchModel(red,green,blue,20*powi(4,level)){
		this->construct(level);
		this->setVertex(r);
	}
	ogSphereModel(int level = 1):ogPatchModel(20*powi(4,level)){ this->construct(level); }
	~ogSphereModel(){ delete sphere; }
	void draw(float x, float y, float z){ ogPatchModel::draw(x,y,z); }
	void draw(float x, float y, float z, float r){
		this->setVertex(r);
		this->draw(x,y,z);
	}
	void draw(float x, float y, float z, float r, unsigned char red, unsigned char green, unsigned char blue){
		this->setColor(red, green, blue);
		this->draw(x,y,z,r);
	}
};

void ogDrawSphere(float x, float y, float z, float r, unsigned char red, unsigned char green, unsigned char blue, int level){
	static ogSphereModel model(level);
	model.draw(x,y,z,r,red,green,blue);
}


#endif //_OGRENDERX_H
