#ifndef _OGGEOMETRY_H
#define _OGGEOMETRY_H

// Copyright (c) OpenMedia Lab., 2003. ver.7/12/2004

#include <stdio.h>
#include <math.h>
#include <float.h>

// double̍ő吔@DBL_MAX
// double̍ŏ@DBL_EPSILON
//const Double PI    = 3.141592654; //~
//const Double PI2   = 6.283185308; //~~2
Double PI    = 3.141592654; //~
Double PI2   = 6.283185308; //~~2

inline Double rad( Double x ) { return x*1.74532925199433E-2; }
inline Double deg( Double x ) { return x*5.72957795130824E1; }

int powi( int x, int y ){ int z = 1; for(int i=0; i<y; i++ ) z *= x; return z; }


Double squareEquation( Double &xSmall, Double &xLarge, Double a, Double b, Double c ){
	Double d, d2;
	if( (d2 = b * b - 4.0 * a * c) < 0.0 ) return d2;
	d = sqrt(d2);
	xSmall = (-b - d)/(2.0 * a);
	xLarge  = (-b + d)/(2.0 * a);
	return d2;
}

class ogVector{
public:
	Double x;
	Double y;
	Double z;

	ogVector( Double a, Double b, Double c ){ x=a; y=b; z=c; }
	ogVector( Double a, Double b ){ x=a; y=b; z=0; }
	ogVector(){}
	Double abs(){ return sqrt(x*x + y*y + z*z); }
	Double abs2(){ return x*x + y*y + z*z; }
	void   add( ogVector &a ){ x+=a.x; y+=a.y; z+=a.z; }
	void   add( ogVector &a, ogVector &b ){ x=a.x+b.x; y=a.y+b.y; z=a.z+b.z; }
	void   add( Double a, Double b, Double c ){ x+=a; y+=b; z+=c; }
	void   affine( class ogMatrix &matrix, ogVector &a );
	void   affine( ogVector &a, class ogMatrix &matrix );
	void   affine( class ogMatrix &matrix ){ ogVector a = *this; this->affine( matrix, a ); }
	Double angle( ogVector &a ){ return acos(a * *this / sqrt(a.abs2() * this->abs2()) ); }
	Double distance( ogVector &a ){ return sqrt((x-a.x)*(x-a.x) + (y-a.y)*(y-a.y) + (z-a.z)*(z-a.z)); }
	Double distance2( ogVector &a ){ return (x-a.x)*(x-a.x) + (y-a.y)*(y-a.y) + (z-a.z)*(z-a.z); }
	int    divide( ogVector &a, ogVector &b, Double m, Double n );
	int    divide( Double m, Double n, ogVector &a, ogVector &b ){ return this->divide( a, b, m, n ); }
	void   exchange( ogVector &a ){ ogVector b = a; a = *this; *this = b; }
	void   exterior( ogVector &a, ogVector &b ){ x=a.y*b.z-b.y*a.z; y=a.z*b.x-b.z*a.x; z=a.x*b.y-b.x*a.y; }
	int    ifZero(){ return this->abs2() < DBL_EPSILON ? 1 : 0; }
	void   intersectionNear( class ogLine &line, class ogSphere &sphere );
	void   intersectionNear( class ogSphere &sphere, class ogLine &line ){ this->intersectionNear( line, sphere ); }
	void   intersectionFar( class ogLine &line, class ogSphere &sphere );
	void   intersectionFar( class ogSphere &sphere, class ogLine &line ){ this->intersectionFar( line, sphere ); }
	void   intersection( class ogLine &line, class ogPlane &plane );
	void   intersection( class ogPlane &plane, class ogLine &line ){ this->intersection( line, plane ); }
	Double latitude(){ return asin( z/sqrt(x*x+y*y) ); }
	Double longitude(){ return atan2( y, x ); }
	void   linear( class ogMatrix &matrix, ogVector &a );
	void   linear( ogVector &a, class ogMatrix &matrix );
	void   linear( class ogMatrix &matrix ){ ogVector a = *this; this->linear( matrix, a ); }
	void   linear( class ogMatrix3 &matrix, ogVector &a );
	void   linear( ogVector &a, class ogMatrix3 &matrix );
	void   linear( class ogMatrix3 &matrix ){ ogVector a = *this; this->linear( matrix, a ); }
	void   multiply( Double k ){ x*=k; y*=k; z*=k; }
	void   multiply( Double k, ogVector &a ){ x=k*a.x; y=k*a.y; z=k*a.z; }
	void   multiply( ogVector &a, Double k ){ this->multiply( k, a ); }
	void   normal( ogVector &a, ogVector &b, ogVector &c ){ ogVector d1, d2; d1.subtract( b, a ); d2.subtract( c, b ); this->exterior( d1, d2 ); }
	void   normalize(){ this->multiply( 1.0 / this->abs() ); }
	void   normalize( ogVector &a ){ this->multiply( 1.0 / a.abs(), a ); }
	void   print(){ printf( "%f %f %f\n", x, y, z ); }
	void   projection( ogVector &a, ogVector &b ){ this->multiply( a*b/b.abs2(), b ); }
	void   projection( ogVector &a ){ this->projection( *this, a ); }
	void   projection( class ogLine &line );
	void   projection( ogVector &a, class ogLine &line ){ *this = a; this->projection( line ); }
	void   projection( class ogLine &line, ogVector &a ){ this->projection( a, line ); }
	void   reverse(){ x = -x; y = -y; z = -z; }
	void   reverse( ogVector &a ){ x = -a.x; y = -a.y; z = -a.z; }
	void   rotate( ogVector &axis, Double angle );
	void   rotate( Double angle, ogVector &axis ){ this->rotate( axis, angle ); }
	void   rotate( ogVector &a, ogVector &axis, Double angle ){ *this = a; this->rotate( axis, angle ); }
	void   rotate( class ogLine &axis, Double angle );
	void   rotate( ogVector &a, class ogLine &axis, Double angle ){ *this = a; this->rotate( axis, angle ); }
	void   rotateX( ogVector &a, Double ang ){ Double c=cos(ang); Double s=sin(ang); x=a.x; y=c*a.y-s*a.z; z=s*a.y+c*a.z; }
	void   rotateX( Double ang, ogVector &a ){ this->rotateX( a, ang ); }
	void   rotateX( Double ang ){ ogVector a; a= *this; this->rotateX( a, ang ); }
	void   rotateY( ogVector &a, Double ang ){ Double c=cos(ang); Double s=sin(ang); y=a.y; z=c*a.z-s*a.x; x=s*a.z+c*a.x; }
	void   rotateY( Double ang, ogVector &a ){ this->rotateY( a, ang ); }
	void   rotateY( Double ang ){ ogVector a; a= *this; this->rotateY( a, ang ); }
	void   rotateZ( ogVector &a, Double ang ){ Double c=cos(ang); Double s=sin(ang); z=a.z; x=c*a.x-s*a.y; y=s*a.x+c*a.y; }
	void   rotateZ( Double ang, ogVector &a ){ this->rotateZ( a, ang ); }
	void   rotateZ( Double ang ){ ogVector a; a= *this; this->rotateZ( a, ang ); }
	void   scale( Double a, Double b, Double c ){ x*=a; y*=b; z*=c; }
	void   set( Double a, Double b, Double c ){ x=a; y=b; z=c; }
	void   set( Double a, Double b ){ x=a; y=b; }
	void   set( class ogVector2 &a );
	void   setPolar( Double lng, Double lat ){ x=cos(lng)*cos(lat); y=sin(lng)*cos(lat); z=sin(lat); }
	void   subtract( ogVector &a ){ x-=a.x; y-=a.y; z-=a.z; }
	void   subtract( ogVector &a, ogVector &b ){ x=a.x-b.x; y=a.y-b.y; z=a.z-b.z; }
	void   subtract( Double a, Double b, Double c ){ x-=a; y-=b; z-=c; }
	void   symmetry( ogVector &a ){ this->reverse(); this->add( a ); this->add( a ); }
	void   symmetry( class ogPlane &plane );
	void   symmetry( class ogPlane &plane, ogVector &a ){ *this = a; this->symmetry( plane ); }
	void   symmetry( ogVector &a, class ogPlane &plane ){ this->symmetry( plane, a ); }
	void   symmetry( class ogLine &line );
	void   symmetry( class ogLine &line, ogVector &a ){ *this = a; this->symmetry( line ); }
	void   symmetry( ogVector &a, class ogLine &line ){ this->symmetry( line, a ); }
	void   unitX(){ this->set( 1.0, 0.0, 0.0 ); }
	void   unitY(){ this->set( 0.0, 1.0, 0.0 ); }
	void   unitZ(){ this->set( 0.0, 0.0, 1.0 ); }
	void   zero(){ this->set( 0.0, 0.0, 0.0 ); }
#ifdef __GL_H__
	void   glNormal(){ glNormal3dv( (GLdouble *)this ); }
	void   glVertex(){ glVertex3dv( (GLdouble *)this ); }
#endif
	ogVector operator +( ogVector &a ){ ogVector b; b.add( *this, a ); return b; }
	ogVector operator +=( ogVector &a ){ this->add( a ); return *this; }
	ogVector operator -( ogVector &a ){ ogVector b; b.subtract( *this, a ); return b; }
	ogVector operator -=( ogVector &a ){ this->subtract( a ); return *this; }
	ogVector operator -(){ ogVector a; a.reverse( *this ); return a; }
	Double  operator *( ogVector &a ){ return x*a.x + y*a.y + z*a.z; }
	ogVector operator *( Double k ){ ogVector a; a.multiply( k, *this ); return a; }
	ogVector operator /( Double k ){ ogVector a; a.multiply( 1.0/k, *this ); return a; }
	friend ogVector operator *( Double k, ogVector &a ){ return a * k; }
	ogVector operator *( class ogMatrix &m ){ ogVector a; a.affine( *this, m ); return a; }
	ogVector operator *( class ogMatrix3 &m ){ ogVector a; a.linear( m, *this ); return a; }
	ogVector operator *=( Double k ){ this->multiply( k ); return *this; }
	ogVector operator /=( Double k ){ this->multiply( 1.0/k ); return *this; }
	ogVector operator *=( class ogMatrix &m ){ this->affine( m ); return *this; }
	ogVector operator &( ogVector &a ){ ogVector b; b.exterior( *this, a ); return b; }
	ogVector operator =( class ogVector2 &a ){ this->set( a ); return *this; }
};

class ogVectorf{
public:
	float x;
	float y;
	float z;
};

int ogVector::divide(  ogVector &a, ogVector &b, Double m, Double n ){
ogVector c;
	if( fabs(m+n) < DBL_EPSILON ){
		printf("Unexpected ratio %f : %f in ogVector::divide()!\n", m, n );
		return 1;
	}
	c.multiply( n, a );
	this->multiply( m, b );
	this->add( c );
	this->multiply( 1.0/(m+n) );
	return 0;
}

void ogVector::rotate( ogVector &axis, Double angle ){
ogVector projected, uaxis, vaxis;
	projected.projection( *this, axis );
	uaxis.subtract( *this, projected );
	vaxis.exterior( axis, uaxis );
	uaxis.multiply( cos( angle ) );
	vaxis.multiply( sin( angle )/axis.abs() );
	this->add( uaxis, vaxis );
	this->add( projected );
}

class ogVector2{
public:
	Double x;
	Double y;

	ogVector2( Double a, Double b ){ x=a; y=b; }
	ogVector2(){}
	Double abs(){ return sqrt(x*x + y*y); }
	Double abs2(){ return x*x + y*y; }
	void   add( ogVector2 &a ){ x+=a.x; y+=a.y; }
	void   add( ogVector2 &a, ogVector2 &b ){ x=a.x+b.x; y=a.y+b.y; }
	void   add( Double a, Double b ){ x+=a; y+=b; }
	void   affine( class ogMatrix3 &m, ogVector2 &a );
	void   affine( ogVector2 &a, class ogMatrix3 &m );
	void   affine( class ogMatrix3 &matrix ){ ogVector2 a = *this; this->affine( matrix, a ); }
	Double angle( ogVector2 &a ){ return acos(a * *this / sqrt(a.abs2() * this->abs2()) ); }
	Double distance( ogVector2 &a ){ return sqrt((x-a.x)*(x-a.x) + (y-a.y)*(y-a.y)); }
	Double distance2( ogVector2 &a ){ return (x-a.x)*(x-a.x) + (y-a.y)*(y-a.y); }
	int    divide( ogVector2 &a, ogVector2 &b, Double m, Double n );
	int    divide( Double m, Double n, ogVector2 &a, ogVector2 &b ){ return this->divide( a, b, m, n ); }
	void   exchange( ogVector2 &a ){ ogVector2 b = a; a = *this; *this = b; }
	int    ifZero(){ return this->abs2() < DBL_EPSILON ? 1 : 0; }
	void   linear( class ogMatrix3 &matrix, ogVector2 &a );
	void   linear( ogVector2 &a, class ogMatrix3 &matrix );
	void   linear( class ogMatrix3 &matrix ){ ogVector2 a = *this; this->linear( matrix, a ); }
	void   linear( class ogMatrix2 &matrix, ogVector2 &a );
	void   linear( ogVector2 &a, class ogMatrix2 &matrix );
	void   linear( class ogMatrix2 &matrix ){ ogVector2 a = *this; this->linear( matrix, a ); }
	Double longitude(){ return atan2( y, x ); }
	void   multiply( Double k ){ x*=k; y*=k; }
	void   multiply( Double k, ogVector2 &a ){ x=k*a.x; y=k*a.y; }
	void   multiply( ogVector2 &a, Double k ){ this->multiply( k, a ); }
	void   normalize(){ this->multiply( 1.0 / this->abs() ); }
	void   normalize( ogVector2 &a ){ this->multiply( 1.0 / a.abs(), a ); }
	void   print(){ printf( "%f %f\n", x, y ); }
	void   projection( ogVector2 &a, ogVector2 &b ){ this->multiply( a*b/b.abs2(), b ); }
	void   reverse(){ x = -x; y = -y; }
	void   reverse( ogVector2 &a ){ x = -a.x; y = -a.y; }
	void   rotate( ogVector2 &a, Double ang ){ Double c=cos(ang); Double s=sin(ang); x=c*a.x-s*a.y; y=s*a.x+c*a.y; }
	void   rotate( Double ang, ogVector2 &a ){ this->rotate( a, ang ); }
	void   rotate( Double ang ){ ogVector2 a; a= *this; this->rotate( a, ang ); }
	void   scale( Double a, Double b ){ x*=a; y*=b; }
	void   set( Double a, Double b ){ x=a; y=b; }
	void   setPolar( Double lng ){ x=cos(lng); y=sin(lng); }
	void   subtract( ogVector2 &a ){ x-=a.x; y-=a.y; }
	void   subtract( ogVector2 &a, ogVector2 &b ){ x=a.x-b.x; y=a.y-b.y; }
	void   subtract( Double a, Double b ){ x-=a; y-=b; }
	void   unitX(){ this->set( 1.0, 0.0 ); }
	void   unitY(){ this->set( 0.0, 1.0 ); }
	void   zero(){ this->set( 0.0, 0.0 ); }
#ifdef __GL_H__
	void   glVertex(){ glVertex2dv( (GLdouble *)this ); }
	void   glTexCoord(){ glTexCoord2dv( (GLdouble *)this ); }
#endif
	Double squareEquation( Double a, Double b, Double c );
	ogVector2 operator +( ogVector2 &a ){ ogVector2 b; b.add( *this, a ); return b; }
	ogVector2 operator +=( ogVector2 &a ){ this->add( a ); return *this; }
	ogVector2 operator -( ogVector2 &a ){ ogVector2 b; b.subtract( *this, a ); return b; }
	ogVector2 operator -=( ogVector2 &a ){ this->subtract( a ); return *this; }
	ogVector2 operator -(){ ogVector2 a; a.reverse( *this ); return a; }
	Double  operator *( ogVector2 &a ){ return x*a.x + y*a.y; }
	ogVector2 operator *( Double k ){ ogVector2 a; a.multiply( k, *this ); return a; }
	ogVector2 operator /( Double k ){ ogVector2 a; a.multiply( 1.0/k, *this ); return a; }
	friend ogVector2 operator *( Double k, ogVector2 &a ){ return a * k; }
	ogVector2 operator *( class ogMatrix3 &m ){ ogVector2 a; a.affine( *this, m ); return a; }
	friend ogVector2 operator *( class ogMatrix3 &m, ogVector2 &a ){ ogVector2 b; b.affine( m, a ); return b; }
	ogVector2 operator *=( Double k ){ this->multiply( k ); return *this; }
	ogVector2 operator /=( Double k ){ this->multiply( 1.0/k ); return *this; }
	ogVector2 operator *=( class ogMatrix3 &m ){ this->affine( m ); return *this; }
	ogVector2 operator =( class ogVector &a ){ x = a.x; y = a.y; return *this; }
};

class ogVectorf2{
public:
	float x;
	float y;
};

int ogVector2::divide(  ogVector2 &a, ogVector2 &b, Double m, Double n ){
	if( fabs(m+n) < DBL_EPSILON ){
		printf("Unexpected ratio %f : %f in ogVector2::divide()!\n", m, n );
		return 1;
	}
	ogVector2 c;
	c.multiply( n, a );
	this->multiply( m, b );
	this->add( c );
	this->multiply( 1.0/(m+n) );
	return 0;
}

Double ogVector2::squareEquation( Double a, Double b, Double c ){
	Double d, d2;
	if((d2 = b * b - 4.0 * a * c) < 0.0) return d2;
	d = sqrt(d2);
	x = (-b - d)/(2.0 * a);
	y  = (-b + d)/(2.0 * a);
	return d2;
}

void ogVector::set( class ogVector2 &a ){ x = a.x; y = a.y; }

int _minv(Double *im,Double *m,int size){
int	i, j, k,j2;
int *num, *pivot ;
Double pv, *a, *ia, atmp, *p1, *p2, *p3;
	num = new int[size];
	pivot = new int[size];
	a  = new Double[size*size];
	ia = new Double[size*size];
	
	for(k = 0, p1=m, p2=a, p3=ia; k < size*size; k++){
		*p2++ = *p1++;
		*p3++ = 0.0;
	}

	for(k = 0; k < size; k++){
		ia[k*size+k] = 1.0;
		num[k] = 1;
	}
	for(k = 0; k < size; k++) { /* forward */
		pv = 0.0;
		for(i = 0; i < size; i++) { /* search pivot */
			if( num[i] ){
				if(fabs(a[i*size+k]) > fabs(pv)){
					pv = a[i*size+k];
					pivot[k] = i;
				}
			 }
		}
		if( fabs(pv) <= DBL_EPSILON ) return 1;
		for(j = k; j < size; j++)  a[pivot[k]*size+j] /= pv;
		for(j = 0; j < size; j++) ia[pivot[k]*size+j] /= pv;
		num[pivot[k]] = 0;
		for(i = 0; i < size ; i++) { /* do other lines */
		    if(num[i]){
			atmp = a[i*size+k];
			for(j = k+1; j < size; j++)
			    a[i*size+j] -=  a[pivot[k]*size+j] * atmp;
			for(j = 0; j < size; j++)
			    ia[i*size+j] -= ia[pivot[k]*size+j] * atmp;
		    }
		}
	}
	for(k = size-2; k >= 0; k--){ /* back */
	    for(j = k+1; j < size; j++){
		atmp = a[pivot[k]*size+j];
		    for(j2 = 0; j2 < size; j2++)
			ia[pivot[k]*size+j2] -= ia[pivot[j]*size+j2] * atmp;
	    }
	}
	for(i = 0; i < size; i++){
		for(j = 0; j < size; j++){
			im[i*size+j]=ia[pivot[i]*size+j];
		}
	}
	delete num;
	delete pivot;
	delete a;
	delete ia;
	return 0;
}

class ogMatrix{
public:
	Double e[16];

	void add( ogMatrix &a ){ for( int i=0; i<16; i++ ) e[i] += a.e[i]; }
	void add( ogMatrix &a, ogMatrix &b ){ for( int i=0; i<16; i++ ) e[i] = a.e[i] + b.e[i]; }
	void exchange( ogMatrix &a ){ ogMatrix b = a; a = *this; *this = b; }
	int  inverse( ogMatrix &m ){ return _minv( e, m.e, 4 ); }
	int  inverse(){ ogMatrix m = *this; return _minv( e, m.e, 4 ); }
	void multiply( Double k ){ for( int i=0; i<16; i++) e[i]*=k; }
	void multiply( Double k, ogMatrix &m ){ for( int i=0; i<16; i++) e[i]=k*m.e[i]; }
	void multiply( ogMatrix &m, Double k ){ this->multiply( k, m ); }
	void multiply( ogMatrix &a, ogMatrix &b ){
		e[0] = a.e[0]*b.e[0] + a.e[1]*b.e[4] + a.e[2]*b.e[8] + a.e[3]*b.e[12];
		e[1] = a.e[0]*b.e[1] + a.e[1]*b.e[5] + a.e[2]*b.e[9] + a.e[3]*b.e[13];
		e[2] = a.e[0]*b.e[2] + a.e[1]*b.e[6] + a.e[2]*b.e[10] + a.e[3]*b.e[14];
		e[3] = a.e[0]*b.e[3] + a.e[1]*b.e[7] + a.e[2]*b.e[11] + a.e[3]*b.e[15];
		e[4] = a.e[4]*b.e[0] + a.e[5]*b.e[4] + a.e[6]*b.e[8] + a.e[7]*b.e[12];
		e[5] = a.e[4]*b.e[1] + a.e[5]*b.e[5] + a.e[6]*b.e[9] + a.e[7]*b.e[13];
		e[6] = a.e[4]*b.e[2] + a.e[5]*b.e[6] + a.e[6]*b.e[10] + a.e[7]*b.e[14];
		e[7] = a.e[4]*b.e[3] + a.e[5]*b.e[7] + a.e[6]*b.e[11] + a.e[7]*b.e[15];
		e[8] = a.e[8]*b.e[0] + a.e[9]*b.e[4] + a.e[10]*b.e[8] + a.e[11]*b.e[12];
		e[9] = a.e[8]*b.e[1] + a.e[9]*b.e[5] + a.e[10]*b.e[9] + a.e[11]*b.e[13];
		e[10] = a.e[8]*b.e[2] + a.e[9]*b.e[6] + a.e[10]*b.e[10] + a.e[11]*b.e[14];
		e[11] = a.e[8]*b.e[3] + a.e[9]*b.e[7] + a.e[10]*b.e[11] + a.e[11]*b.e[15];
		e[12] = a.e[12]*b.e[0] + a.e[13]*b.e[4] + a.e[14]*b.e[8] + a.e[15]*b.e[12];
		e[13] = a.e[12]*b.e[1] + a.e[13]*b.e[5] + a.e[14]*b.e[9] + a.e[15]*b.e[13];
		e[14] = a.e[12]*b.e[2] + a.e[13]*b.e[6] + a.e[14]*b.e[10] + a.e[15]*b.e[14];
		e[15] = a.e[12]*b.e[3] + a.e[13]*b.e[7] + a.e[14]*b.e[11] + a.e[15]*b.e[15];
	}
	void print(){ for( int i=0; i<4; i++ ) printf( "%f %f %f %f\n", e[i*4], e[i*4+1], e[i*4+2], e[i*4+3] ); }
	void reverse(){ for( int i=0; i<16; i++ ) e[i] = -e[i]; }
	void reverse( ogMatrix &a ){ for( int i=0; i<16; i++ ) e[i] = -a.e[i]; }
	void set( Double a1, Double a2, Double a3, Double a4, Double b1, Double b2, Double b3, Double b4,
		Double c1, Double c2, Double c3, Double c4, Double d1, Double d2, Double d3, Double d4 ){
		e[0]=a1; e[1]=a2; e[2]=a3; e[3]=a4; e[4]=b1; e[5]=b2; e[6]=b3; e[7]=b4;
		e[8]=c1; e[9]=c2; e[10]=c3; e[11]=c4; e[12]=d1; e[13]=d2; e[14]=d3; e[15]=d4; }
	void set( Double a1, Double a2, Double a3, Double b1, Double b2, Double b3, Double c1, Double c2, Double c3 ){
		e[0]=a1; e[1]=a2; e[2]=a3; e[4]=b1; e[5]=b2; e[6]=b3; e[8]=c1; e[9]=c2; e[10]=c3; }
	void setColumn( int col, Double a, Double b, Double c, Double d ){ e[col]=a; e[col+4]=b; e[col+8]=c; e[col+12]=d; }
	void setRow( int row, Double a, Double b, Double c, Double d ){ e[row*4]=a; e[row*4+1]=b; e[row*4+2]=c; e[row*4+3]=d; }
	void subtract( ogMatrix &a ){ for( int i=0; i<16; i++ ) e[i] -= a.e[i]; }
	void subtract( ogMatrix &a, ogMatrix &b ){ for( int i=0; i<16; i++ ) e[i] = a.e[i] - b.e[i]; }
	void exchange( Double &a, Double &b ){ Double c = a; a = b; b = c; }
	void transpose(){ this->exchange( e[1], e[4] ); this->exchange( e[2], e[8] ); this->exchange( e[3], e[12] );
		this->exchange( e[6], e[9] ); this->exchange( e[7], e[13] ); this->exchange( e[11], e[14] ); }
	void transpose( ogMatrix &m ){ *this = m; this->transpose(); }
	void unit(){ for( int i=0; i<16; i++) e[i] = i%5==0 ? 1.0 : 0.0; }
	void zero(){ for( int i=0; i<16; i++) e[i] = 0.0; }
	ogMatrix(){};
	ogMatrix( Double a1, Double a2, Double a3, Double a4, Double b1, Double b2, Double b3, Double b4,
		Double c1, Double c2, Double c3, Double c4, Double d1, Double d2, Double d3, Double d4 )
		{ this->set( a1, a2, a3, a4, b1, b2, b3, b4, c1, c2, c3, c4, d1, d2, d3, d4 ); }
	ogMatrix( Double a1, Double a2, Double a3, Double b1, Double b2, Double b3, Double c1, Double c2, Double c3 )
		{ this->set( a1, a2, a3, b1, b2, b3, c1, c2, c3 ); }
	ogMatrix( Double *a ){ e[0]=a[0]; e[1]=a[1]; e[2]=a[2]; e[4]=a[3]; e[5]=a[4]; e[6]=a[5]; e[8]=a[6]; e[9]=a[7]; e[10]=a[8]; }
	ogMatrix operator +( ogMatrix &a ){ ogMatrix b; b.add( *this, a ); return b; }
	ogMatrix operator +=( ogMatrix &a ){ this->add( a ); return *this; }
	ogMatrix operator -( ogMatrix &a ){ ogMatrix b; b.subtract( *this, a ); return b; }
	ogMatrix operator -=( ogMatrix &a ){ this->subtract( a ); return *this; }
	ogMatrix operator -(){ ogMatrix a; a.reverse( *this ); return a; }
	ogMatrix operator *( Double k ){ ogMatrix a; a.multiply( k, *this ); return a; }
	ogMatrix operator *( ogMatrix &a ){ ogMatrix b; b.multiply( *this, a ); return b; }
	ogMatrix operator *=( Double k ){ this->multiply( k ); return *this; }
	ogMatrix operator /=( Double k ){ this->multiply( 1.0/k ); return *this; }
	friend ogMatrix operator *( Double k, ogMatrix &a ){ return a * k; }
	ogVector operator *( ogVector &a ){ ogVector b; b.affine( *this, a ); return b; }
};

void ogVector::affine( class ogMatrix &matrix, ogVector &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1]+a.z*matrix.e[2]+matrix.e[3];
	y = a.x*matrix.e[4]+a.y*matrix.e[5]+a.z*matrix.e[6]+matrix.e[7];
	z = a.x*matrix.e[8]+a.y*matrix.e[9]+a.z*matrix.e[10]+matrix.e[11];
}

void ogVector::affine( ogVector &a, class ogMatrix &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[4]+a.z*matrix.e[8]+matrix.e[12];
	y = a.x*matrix.e[1]+a.y*matrix.e[5]+a.z*matrix.e[9]+matrix.e[13];
	z = a.x*matrix.e[2]+a.y*matrix.e[6]+a.z*matrix.e[10]+matrix.e[14];
}

void ogVector::linear( class ogMatrix &matrix, ogVector &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1]+a.z*matrix.e[2];
	y = a.x*matrix.e[4]+a.y*matrix.e[5]+a.z*matrix.e[6];
	z = a.x*matrix.e[8]+a.y*matrix.e[9]+a.z*matrix.e[10];
}

void ogVector::linear( ogVector &a, class ogMatrix &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[4]+a.z*matrix.e[8];
	y = a.x*matrix.e[1]+a.y*matrix.e[5]+a.z*matrix.e[9];
	z = a.x*matrix.e[2]+a.y*matrix.e[6]+a.z*matrix.e[10];
}

class ogMatrix3{
public:
	Double e[9];

	void add( ogMatrix3 &a ){ for( int i=0; i<9; i++ ) e[i] += a.e[i]; }
	void add( ogMatrix3 &a, ogMatrix3 &b ){ for( int i=0; i<9; i++ ) e[i] = a.e[i] + b.e[i]; }
	void exchange( ogMatrix3 &a ){ ogMatrix3 b = a; a = *this; *this = b; }
	int  inverse( ogMatrix3 &m ){ return _minv( e, m.e, 3 ); }
	int  inverse(){ ogMatrix3 m = *this; return _minv( e, m.e, 3 ); }
	void multiply( Double k ){ for( int i=0; i<9; i++) e[i]*=k; }
	void multiply( Double k, ogMatrix3 &a ){ for( int i=0; i<9; i++) e[i]=k*a.e[i]; }
	void multiply( ogMatrix3 &a, ogMatrix3 &b ){
		e[0] = a.e[0]*b.e[0] + a.e[1]*b.e[3] + a.e[2]*b.e[6];
		e[1] = a.e[0]*b.e[1] + a.e[1]*b.e[4] + a.e[2]*b.e[7];
		e[2] = a.e[0]*b.e[2] + a.e[1]*b.e[5] + a.e[2]*b.e[8];
		e[3] = a.e[3]*b.e[0] + a.e[4]*b.e[3] + a.e[5]*b.e[6];
		e[4] = a.e[3]*b.e[1] + a.e[4]*b.e[4] + a.e[5]*b.e[7];
		e[5] = a.e[3]*b.e[2] + a.e[4]*b.e[5] + a.e[5]*b.e[8];
		e[6] = a.e[6]*b.e[0] + a.e[7]*b.e[3] + a.e[8]*b.e[6];
		e[7] = a.e[6]*b.e[1] + a.e[7]*b.e[4] + a.e[8]*b.e[7];
		e[8] = a.e[6]*b.e[2] + a.e[7]*b.e[5] + a.e[8]*b.e[8];
	}
	void print(){ for( int i=0; i<3; i++ ) printf( "%f %f %f\n", e[i*3], e[i*3+1], e[i*3+2] ); }
	void reverse(){ for( int i=0; i<9; i++ ) e[i] = -e[i]; }
	void reverse( ogMatrix3 &a ){ for( int i=0; i<9; i++ ) e[i] = -a.e[i]; }
	void set( Double a1, Double a2, Double a3, Double b1, Double b2, Double b3, Double c1, Double c2, Double c3 ){
		e[0]=a1; e[1]=a2; e[2]=a3; e[3]=b1; e[4]=b2; e[5]=b3; e[6]=c1; e[7]=c2; e[8]=c3; }
	void set( Double a1, Double a2, Double b1, Double b2 ){ e[0]=a1; e[1]=a2; e[3]=b1; e[4]=b2; }
	void setColumn( int col, Double a, Double b, Double c ){ e[col]=a; e[col+3]=b; e[col+6]=c; }
	void setRow( int row, Double a, Double b, Double c ){ e[row*3]=a; e[row*3+1]=b; e[row*3+2]=c; }
	void subtract( ogMatrix3 &a ){ for( int i=0; i<9; i++ ) e[i] -= a.e[i]; }
	void subtract( ogMatrix3 &a, ogMatrix3 &b ){ for( int i=0; i<9; i++ ) e[i] = a.e[i] - b.e[i]; }
	void exchange( Double &a, Double &b ){ Double c = a; a = b; b = c; }
	void transpose(){ this->exchange( e[1], e[3] ); this->exchange( e[2], e[6] ); this->exchange( e[5], e[7] ); }
	void unit(){ for( int i=0; i<9; i++) e[i] = i%4==0 ? 1.0 : 0.0; }
	void zero(){ for( int i=0; i<9; i++) e[i] = 0.0; }
	ogMatrix3(){};
	ogMatrix3( Double a1, Double a2, Double a3, Double b1, Double b2, Double b3, Double c1, Double c2, Double c3 )
		{ this->set( a1, a2, a3, b1, b2, b3, c1, c2, c3 ); }
	ogMatrix3( Double a1, Double a2, Double b1, Double b2 ){ this->set( a1, a2, b1, b2 ); }
	ogMatrix3( Double *a ){ for( int i=0; i<9; i++) e[i] = a[i]; }
	ogMatrix3 operator +( ogMatrix3 &a ){ ogMatrix3 b; b.add( *this, a ); return b; }
	ogMatrix3 operator +=( ogMatrix3 &a ){ this->add( a ); return *this; }
	ogMatrix3 operator -( ogMatrix3 &a ){ ogMatrix3 b; b.subtract( *this, a ); return b; }
	ogMatrix3 operator -=( ogMatrix3 &a ){ this->subtract( a ); return *this; }
	ogMatrix3 operator -(){ ogMatrix3 a; a.reverse( *this ); return a; }
	ogMatrix3 operator *( Double k ){ ogMatrix3 a; a.multiply( k, *this ); return a; }
	ogMatrix3 operator *( ogMatrix3 &a){ ogMatrix3 b; b.multiply( *this, a ); return b; }
	ogMatrix3 operator *=( Double k ){ this->multiply( k ); return *this; }
	ogMatrix3 operator /=( Double k ){ this->multiply( 1.0/k ); return *this; }
	friend ogMatrix3 operator *( Double k, ogMatrix3 &a ){ return a * k; }
	ogVector operator *( ogVector &a ){ ogVector b; b.linear( *this, a ); return b; }
	ogVector2 operator *( ogVector2 &a ){ ogVector2 b; b.affine( *this, a ); return b; }

};

void ogVector::linear( class ogMatrix3 &matrix, ogVector &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1]+a.z*matrix.e[2];
	y = a.x*matrix.e[3]+a.y*matrix.e[4]+a.z*matrix.e[5];
	z = a.x*matrix.e[6]+a.y*matrix.e[7]+a.z*matrix.e[8];
}

void ogVector::linear( ogVector &a, class ogMatrix3 &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[3]+a.z*matrix.e[6];
	y = a.x*matrix.e[1]+a.y*matrix.e[4]+a.z*matrix.e[7];
	z = a.x*matrix.e[2]+a.y*matrix.e[5]+a.z*matrix.e[8];
}

void ogVector2::affine( class ogMatrix3 &matrix, ogVector2 &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1]+matrix.e[2];
	y = a.x*matrix.e[3]+a.y*matrix.e[4]+matrix.e[5];
}

void ogVector2::affine( ogVector2 &a, class ogMatrix3 &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[3]+matrix.e[6];
	y = a.x*matrix.e[1]+a.y*matrix.e[4]+matrix.e[7];
}

void ogVector2::linear( class ogMatrix3 &matrix, ogVector2 &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1];
	y = a.x*matrix.e[3]+a.y*matrix.e[4];
}

void ogVector2::linear( ogVector2 &a, class ogMatrix3 &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[3];
	y = a.x*matrix.e[1]+a.y*matrix.e[4];
}

class ogMatrix2{
public:
	Double e[4];

	void exchange( Double &a, Double &b ){ Double c = a; a = b; b = c; }
	int  inverse( ogMatrix2 &m ){ return _minv( e, m.e, 2 ); }
	int  inverse(){ ogMatrix2 m = *this; return _minv( e, m.e, 2 ); }
	void multiply( Double k ){ for( int i=0; i<4; i++) e[i]*=k; }
	void multiply( Double k, ogMatrix2 &a ){ for( int i=0; i<4; i++) e[i]=k*a.e[i]; }
	void multiply( ogMatrix2 &a, ogMatrix2 &b ){
		e[0] = a.e[0]*b.e[0] + a.e[1]*b.e[2];
		e[1] = a.e[0]*b.e[1] + a.e[1]*b.e[3];
		e[2] = a.e[2]*b.e[0] + a.e[3]*b.e[2];
		e[3] = a.e[2]*b.e[1] + a.e[3]*b.e[3];
	}
	void print(){ for( int i=0; i<2; i++ ) printf( "%f %f\n", e[i*2], e[i*2+1] ); }
	void set( Double a1, Double a2, Double b1, Double b2 ){ e[0]=a1; e[1]=a2; e[2]=b1; e[3]=b2; }
	void transpose(){ this->exchange( e[1], e[2] ); }
	void unit(){ for( int i=0; i<4; i++) e[i] = i%3==0 ? 1.0 : 0.0; }
	void zero(){ for( int i=0; i<4; i++) e[i] = 0.0; }
	ogMatrix2(){};
	ogMatrix2( Double a1, Double a2, Double b1, Double b2 ){ this->set( a1, a2, b1, b2 ); }
	ogMatrix2( Double *a ){ for( int i=0; i<4; i++) e[i] = a[i]; }
	ogMatrix2 operator *( Double k ){ ogMatrix2 a; a.multiply( k, *this ); return a; }
	ogMatrix2 operator *( ogMatrix2 a ){ ogMatrix2 b = *this; this->multiply( b, a ); return *this; }
	friend ogMatrix2 operator *( Double k, ogMatrix2 &a ){ return a * k; }
	ogVector2 operator *( ogVector2 &a ){ ogVector2 b; b.linear( *this, a ); return b; }
};

void ogVector2::linear( class ogMatrix2 &matrix, ogVector2 &a ){
	x = a.x*matrix.e[0]+a.y*matrix.e[1];
	y = a.x*matrix.e[2]+a.y*matrix.e[3];
}

void ogVector2::linear( ogVector2 &a, class ogMatrix2 &matrix ){
	x = a.x*matrix.e[0]+a.y*matrix.e[2];
	y = a.x*matrix.e[1]+a.y*matrix.e[3];
}

class ogPlane{
public:
	ogVector normal;
	ogVector passage;

	Double distance( ogVector &point ){ return fabs( this->signedDistance( point ) ); }
	void equidistance( ogVector &a, ogVector &b ){
		normal = b - a;
		passage = a + b;
		passage.multiply( 0.5 );
	}
	void print(){ printf( "normal " ); normal.print(); printf( "passage " ); passage.print(); }
	ogPlane set( Double nx, Double ny, Double nz,  Double px, Double py, Double pz ){ normal.set( nx, ny, nz ); passage.set( px, py, pz ); return *this; }
	ogPlane set( Double a, Double b, Double c, Double d ){
		normal.x = a; normal.y = b; normal.z = c;
		if( fabs( a ) > fabs( b ) && fabs( a ) > fabs( c ) ){
			passage.set( -d / a, 0.0, 0.0 );
		}else if( fabs( b ) > fabs( c ) ){
			passage.set( 0.0, -d / b, 0.0 );
		}else{
			passage.set( 0.0, 0.0, -d / c );
		}
		return *this;
	}
	ogPlane set( ogVector &normal, ogVector &passage ){ this->normal = normal; this->passage = passage; }
	Double signedDistance( ogVector &point ){ return ( point - passage ) * normal; }
	ogPlane(){};
	ogPlane ( Double nx, Double ny, Double nz,  Double px, Double py, Double pz ){ this->set( nx, ny, nz, px, py, pz ); }
	ogPlane( Double a, Double b, Double c, Double d ){ this->set( a, b, c, d ); }
	ogPlane( ogVector &normal, ogVector &passage ){ this->normal = normal; this->passage = passage; }
};

void ogVector::symmetry( ogPlane &plane ){
	Double t = 2 * ( ( plane.passage - *this ) * plane.normal ) / plane.normal.abs2();
	ogVector v = t * plane.normal;
	*this += v;
}

class ogLine{
public:
	ogVector direction;
	ogVector passage;

	Double distance( ogVector &point ){
	ogPlane plane;
		plane.normal = direction;
		plane.passage = point;
		ogVector projected;
		projected.intersection( *this, plane );
		return projected.distance( point );
	}
	void equidistance( ogLine &a, ogLine &b ){
		ogVector vertical;
			vertical.exterior( a.direction, b.direction );
			ogPlane pl;
			pl.normal.exterior( a.direction, vertical );
			pl.passage = a.passage;
			a.direction.normalize();
			b.direction.normalize();
			direction = a.direction + b.direction;
			passage.intersection( pl, b );
	}
	void intersection( ogPlane &a, ogPlane &b ){
		Double ai, bi, d;
		direction.exterior( a.normal, b.normal );
		direction.normalize();
		ai = a.normal * a.passage;
		bi = b.normal * b.passage;
		if( fabs( direction.x ) > fabs( direction.y ) && fabs( direction.x ) > fabs( direction.z ) ){
			passage.x = 0;
			d  = a.normal.y * b.normal.z - a.normal.z * b.normal.y;
			passage.y = ( ai * b.normal.z - bi * a.normal.z) / d;
			passage.z = -( ai * b.normal.y - bi * a.normal.y) / d;
		}else if( fabs( direction.y ) > fabs( direction.z ) ){
			passage.y = 0;
			d  = a.normal.z * b.normal.x - a.normal.x * b.normal.z;
			passage.z = ( ai * b.normal.x - bi * a.normal.x) / d;
			passage.x = -( ai * b.normal.z - bi * a.normal.z) / d;
		}else{
			passage.z = 0;
			d  = a.normal.x * b.normal.y - a.normal.y * b.normal.x;
			passage.x = ( ai * b.normal.y - bi * a.normal.y) / d;
			passage.y = -( ai * b.normal.x - bi * a.normal.x) / d;
		}
	}
	void print(){ printf( "direction " ); direction.print(); printf( "passage " ); passage.print(); }
	void set( ogVector &a, ogVector &b ){ direction.subtract( b, a ); passage = a; }
	ogLine(){};
	ogLine( ogVector &a, ogVector &b ){ this->set( a, b ); }
};

void ogVector::intersection( class ogLine &line, class ogPlane &plane ){
	Double t = ( plane.passage - line.passage ) * plane.normal;
	t /= plane.normal * line.direction;
	*this = line.direction;
	this->multiply( t );
	this->add( line.passage );
}

void ogVector::projection( ogLine &line ){
	this->subtract( line.passage );
	this->projection( line.direction );
	this->add( line.passage );
}

void ogVector::rotate( ogLine &axis, Double angle ){
	this->subtract( axis.passage );
	this->rotate( axis.direction, angle );
	this->add( axis.passage );
}

void ogVector::symmetry( ogLine &line ){
	ogVector a;
	a.projection( *this, line );
	a.subtract( *this );
	a.multiply( 2.0 );
	this->add( a );
}

class ogLine2{
public:
	ogVector2 normal;
	ogVector2 direction; // normal and direction are vertical to each other.
	ogVector2 passage;

	void equidistance( ogVector2 &a, ogVector2 &b ){
		normal = b - a;
		direction.x = normal.y; direction.y = -normal.x;
		passage = a + b;
		passage.multiply( 0.5 );
	}
	ogLine2 set( Double nx, Double ny, Double px, Double py ){ normal.set( nx, ny ); direction.set( ny, -nx ); passage.set( px, py ); return *this; }
	ogLine2 set( ogVector2 &a, ogVector2 &b ){ ogVector2 c = b - a; this->set(c.y, -c.x, a.x, a.y); return *this; }
	ogLine2 set( Double a, Double b, Double c ){
		normal.x = a; normal.y = b;
		direction.x = normal.y; direction.y = -normal.x;
		if( fabs( a ) > fabs( b ) ){
			passage.set( -c / a, 0.0 );
		}else{
			passage.set( 0.0, -c / b );
		}
		return *this;
	}
	Double signedDistance( ogVector2 &a ){ return ( a - passage ) * normal; }
	Double distance( ogVector2 &point ){ return fabs( this->signedDistance( point ) ); }
	void print(){ printf( "normal " ); normal.print(); printf( "passage " ); passage.print(); }
	ogLine2(){};
	ogLine2( Double nx, Double ny, Double px, Double py ){ this->set( nx, ny, px, py ); }
	ogLine2( ogVector2 &a, ogVector2 &b ){ this->set( a, b ); }
	ogLine2( Double a, Double b, Double c ){ this->set( a, b, c ); }
};

class ogSphere{
public:
	ogVector center;
	Double  radius;

	void set( ogVector &center, Double radius ){ this->center = center; this->radius = radius; }
	void set( Double cx, Double cy, Double cz, Double radius ){ center.set(cx,cy,cz); this->radius = radius; }
	ogSphere( ogVector &center, Double radius ){ this->set( center, radius ); }
	ogSphere( Double cx, Double cy, Double cz, Double radius ){ this->set(cx,cy,cz,radius); }
	void print(){ printf( "center " ); center.print(); printf( "radius %f\n", radius ); }
};

void ogVector::intersectionNear( ogLine &line, ogSphere &sphere ){
ogVector vec;
Double a, b, c;
	a = line.direction.abs2();
	vec.subtract( line.passage, sphere.center );
	b = line.direction * vec * 2;
	c = vec.abs2() - sphere.radius * sphere.radius;
	ogVector2 p;
	p.squareEquation( a, b, c );
	this->multiply( line.direction, p.x );
	this->add( line.passage );
}

void ogVector::intersectionFar( ogLine &line, ogSphere &sphere ){
ogVector vec;
Double a, b, c;
	a = line.direction.abs2();
	vec.subtract( line.passage, sphere.center );
	b = line.direction * vec * 2;
	c = vec.abs2() - sphere.radius * sphere.radius;
	ogVector2 p;
	p.squareEquation( a, b, c );
	this->multiply( line.direction, p.y );
	this->add( line.passage );
}

#endif //_OGGEOMETRY_H
